 

/*********************************************************************************************
 ***                                                                                       ***
 ***                  Examples  : RTC+TEMP(DS3232) For "CP-JR ARM7LPC2138                  ***                                                           
 ***                            : Interface I2C : (Use I/O PIN)                            ***
 ***                                                                                       ***
 *********************************************************************************************
 ***                                                                                       ***
 ***  Target MCU : LPC2138                                                                 ***
 ***       	     : X-TAL : 19.6608 MHz                                                     *** 
 ***             : PLL Setup = M(3),P(2)	                               	               ***
 ***             : Run Speed 58.9824 MHz(With PLL)(19.6608MHz x 3 = 58.9824 MHz            ***         
 ***             : VPB Clock(pclk) = CPU Clock/2 = 58.9824MHz/2 = 29.4912 MHz              ***
 ***             : CPU Clock(cclk) = 58.9824 MHz                                           ***           
 ***                                                                                       ***
 *** Editor           : uVision3 (V3.60)                                                   ***
 *** Compiler         : RealView MDK-ARM 3.20                                              ***
 ***																		               ***
 ***  Create By       : Mr. Sittiphol Yooyod (WWW.ETT.CO.TH)                               ***
 ***  Last Update     : 23/December/2011                                                   ***
 ***                                                                                       ***
 ***  Setup DS3232    : Interface I2C(ID:1101000X = 0xD0(Wr),0xD1(Rd) )                    *** 
 ***	                                                                                   ***
 ***  Port Interface  :  									                               ***
 ***             LCD  : P1.25       = RS(OUT)							                   ***	   
 ***                    P1.26       = RW(OUT)								               ***	   
 ***					P1.27       = EN(OUT)								               ***
 ***                    P1.28-P1.31 = Data 4 bit High (OUT)		  		                   ***
 ***                                                                                       ***                                                                                                        **
 ***       5V                                                                              ***
 ***      --+--                                                                            ***
 ***        |                                                                              ***
 ***        +-----------------+                                                            ***
 ***        |                 |                                                            ***
 ***       ---                |                                                            ***
 ***       |\|                |                                                            ***
 ***    VR |/|<---------------|--+                                                         ***
 ***   10K ---                |  | P1.25    1.27          P1.28   P1.30                    ***
 ***        |                 |  |   ^ P1.26 ^              ^       ^                      ***
 ***        |                 |  |   |   ^   |              | P1.29 | P1.31                ***
 ***        +--------------+--|--|---|---|---|--+--+--+--+  |   ^   |   ^                  ***
 ***      __|__            |  |  |   |   |   |  |  |  |  |  |   |   |   |                  ***
 ***       ---             |  |  |   |   |   |  |  |  |  |  |   |   |   |                  ***
 ***        -             1| 2| 3|  4|  5| 6 | 7| 8| 9|10|11|12 | 13| 14|                  ***
 ***                     __|__|__|___|___|___|__|__|__|__|__|___|___|___|_                 ***
 ***                    |  G Vcc Vo RS  R/W  E  D0 D1 D2 D3 D4  D5  D6 D7 |                ***
 ***                    |    _________________________________________    |                ***
 ***                    |   |                                         |   |                ***
 ***                    |   |               Module                    |   |                ***
 ***                    |   |         LCD 2x16 Charecter              |   |                ***  
 ***                    |   |_________________________________________|   |                ***
 ***                    |                                                 |                ***
 ***                    |_________________________________________________|                ***
 ***                                                                                       ***       
 ***																					   ***
 ***   																				       ***
 ***    I2C Interface  :   <MCU LPC2138>	  	  	           <DS3232 >                   *** 
 ***                                                            				           ***
 ***                     P0.17 : SCK(out)       Connect          SCL                       ***
 ***                     P0.18 : SDA(In/Out)    Connect          SDA                       ***  
 ***                                                                                       ***
 ***                                                                                       ***
 *********************************************************************************************/


#include "LPC213x.h" 	                             
#include "stdio.h"


//************** Define LCD Control Pin *******************

#define     DATA_LCD     IOPIN1                 //Referent DATA_LCD = IOPIN1 for P1.28-P1.31(Output)  

#define     RS_HI()      IOSET1 = 0x02000000    //P1.25 = RS:1
#define     RS_LO()      IOCLR1 = 0x02000000    //P1.25 = RS:0


#define     RW_HI()      IOSET1 = 0x04000000    //P1.26 = RW:1
#define     RW_LO()      IOCLR1 = 0x04000000    //P1.26 = RW:0


#define     ENA_HI()     IOSET1 = 0x08000000    //P1.27 = ENA:1
#define     ENA_LO()     IOCLR1 = 0x08000000    //P1.27 = ENA:0


//************** Define I2C Pin **************************


#define     SCL_HI()    IOSET0 = 0x00020000     //P0.17=SCK:1
#define     SCL_LO()	IOCLR0 = 0x00020000     //P0.17=SCK:0

  
#define     SDA_HI()    IOSET0 = 0x00040000     //P0.18=SDA:1
#define     SDA_LO()	IOCLR0 = 0x00040000     //P0.18=SDA:0

#define     SDA_IN()    IODIR0 &= 0xFFFBFFFF    //P0.18=Input          	  
#define     SDA_OUT()   IODIR0 |= 0x00040000    //P0.18=Output           	  

#define     SDA         IOPIN0				    //Referent SDA = IOPIN0 for P0.18(input)




unsigned int ss,mn,hh,tem,dot,dd,mm,yy     ;
unsigned char sign                         ;


/**********************************************************
 **                    Function Delay                    **
 **********************************************************/

  void delay(int cnt)
   {
     int i,j ;
      for(i=0;i<cnt;i++)
        for(j=0;j<cnt;j++) ;
   }

/***********************************************************
 **                     Initial Port                      **
 ***********************************************************/
 
 void init_port(void)
  {
  //------ Set GPIO  PIN -------

  PINSEL1 = 0x00000000               ;     //Set Port0-PIN[P16..P31]= GPIO for I2C
  PINSEL2 = 0x00000000               ;     //Set Port1-PIN[P16..P31]= GPIO for LCD
 
 //------ Set Direction of GPIO PIN -------

  IODIR0  = 0xFFFFFFFF               ;	   //Set Port0-[P0..P31]=Output	for I2C
  IODIR1  = 0xFFFFFFFF               ;	   //Set Port1-[P16..P31]=Output for LCD
 
 //--------- Default Value Port0 ---------

  IOPIN0  = 0x00000000               ;     //Set Data Port0 = 0 For I2C
  IOPIN1  = 0x00000000               ;     //Set Data Port1 = 0 For LCD
 
  } 


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  +                                                                                      +
  +                              LCD Function                                            +
  +                                                                                      +
  +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/


/**********************************
 *    write instruction LCD       *
 **********************************/

void write_cmm(unsigned long cmm)
{
  
  DATA_LCD =  (cmm << 24)& 0xF0000000  ;   //Sent cmm 4 bit(P1.28-P1.31) High,RS:P1.25 = 0,  RW:P1.26=0 
  ENA_HI()                             ;    
  delay(45)                           ;
  ENA_LO()                             ;    
  delay(45)                           ;

  DATA_LCD = (cmm << 28) & 0xF0000000  ;   //Sent cmm 4 bit(P1.28-P1.31) Low,RS:P1.25= 0 ,RW:P1.26=0 
  ENA_HI()                             ;   
  delay(45)                           ;
  ENA_LO()                             ;   
  delay(45)                           ;
}



/*********************************
 *       write data lcd          *
 *********************************/

void write_data(unsigned long dat)
{ 
  DATA_LCD = ((dat << 24) & 0xF0000000)|0x02000000   ;   //Sent data 4 bit High,RS:P1.25 = 1 ,RW:P1.26=0 
  ENA_HI()                                           ;    
  delay(45)                                         ;
  ENA_LO()                                           ;    
  delay(45)                                         ;

  DATA_LCD = ((dat << 28)& 0xF0000000) | 0x02000000   ;   //Sent data 4 bit Low,RS:P1.25 = 1 ,RW:P1.26=0 
  ENA_HI()                                           ;   
  delay(45)                                         ;					    
  ENA_LO()                                           ;   
  delay(45)                                         ;
}

/******************************
 *      Clear Display LCD     *
 ******************************/

void clr_display(void)
 {
  write_cmm(0x01) ;
 }


/**********************************
 *          initial lcd           *
 **********************************/
void init_lcd(void)
{
  write_cmm(0x33);	        //Command Control 
  write_cmm(0x32);
  write_cmm(0x28);
  write_cmm(0x0C);  	    //Set Display On/Off:Display On,Not Show Cursor and not blink Cursor
  write_cmm(0x06);  	    //Set Entry Mode :Increment Address,Cursor Shift Right
    
  write_cmm(0x1C);          //Set Cursor Or Display Shift : Shift Right Charecter
  write_cmm(0x28);      	//Function Set: set interface data 4bit,Set Display LCD 2 Line ,Set Charecter Size 5x7
  write_cmm(0x80|0x00);		//Set Address Begin 0x00 (Line1 Charecter1)
  clr_display();
}



/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  +                                                                                      +
  +                              I2C Function                                            +
  +                                                                                      +
  +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/


 /**********************************************
  **                 I2C Start                **
  **********************************************/

void I2C_Start(void)
{
   
   SDA_OUT()              ;  //Set PA0 = output  
 
   SDA_HI()               ;	
   SCL_HI()               ;
   SDA_LO()               ;	 
   SCL_LO()               ;
}


 /**********************************************
  **                 I2C Stop                **
  **********************************************/

void I2C_Stop(void)
{

   SDA_OUT()             ;  //Set PA0=Output      
  
   SDA_LO()              ;  
   SCL_HI()              ;	  
   SDA_HI()              ;
   
}


/***********************************************************************************
 **                       Function I2C Write  Data 8 bit                          **
 **                                                                               **
 ***********************************************************************************
 **  Parameter :             												      **
 **              dat = Data for write 1Byte                                       **       
 ***********************************************************************************/


void I2C_WrByte(unsigned char dat)			  	
{
   unsigned char loop ;		


   for(loop=0;loop<8;loop++)                  //Loop Write data 8 Bit 
    {			    		
		//--------- write Data(SDA:PA0) ---------

	  SDA_OUT() ;                            //Set PA0=SDA= Output  

	  if((dat & 0x80)== 0x80)	            //Check data bit8 is 0 or 1
		    SDA_HI()                     ;  //Sent '1' Out Pin SDA(PA0)
	  else
		    SDA_LO()                     ;	//Sent '0' Out Pin SDA(PA0)

      dat <<= 1                          ;	//Shift data Next Bit 

	  SCL_HI()                           ;  //Clock HI       
	  SCL_LO()                           ;  //Clock LO       
    } 
	  	  

   SDA_IN()                              ;  //Set PA0=Input 

   SCL_HI()                              ;  //Start ACK Clock
   while(SDA & 0x00040000){;}               //Check ACK:P0.18 = 0 Out Loop
   SCL_LO()                              ;  //End ACK Clock
   	 
}

/************************************************************************ 
 **                   Function Write RTC DS3232                        **
 ** Parameter :                                                        **
 **            addr  = Register Address for write(1Byte)               **
 **            dat   = Data for Set up DS3232(1Byte)                   **    
 ************************************************************************/
 
void I2C_WrRTC (unsigned char addr,unsigned char dat)
{ 
  
   I2C_Start() ;

//---------- Sent Slave address for Write ---------------

   I2C_WrByte(0xD0)       ;  //Send ID Code DS3232+Write (1101 000+0)
 
//------------ Sent address Register --------------
 
   I2C_WrByte(addr)       ;	//Send Address Reg. to DS3232

//------------- Sent data to RTC ---------------

   I2C_WrByte(dat)        ;	//Send Data Control to DS3232
  
   I2C_Stop()              ;
 
 
}


 
/*******************************************************************************
 **                 Function I2C Read  Data 8 bit                             **
 **                                                                           **           
 *******************************************************************************/

 signed char I2C_RdByte(void)			  	
 { 
   signed char loop,result=0 ;		

   for(loop = 0; loop < 8; loop++)	            //Loop Read data 8-Bit
    {
      result <<= 1                          ;   //Shift Result Save (MSB <- LSB)
    
      SDA_IN()                              ;      
      SCL_HI()                              ;   //Strobe Read SDA
   	  	  
	  if((SDA & 0x00040000) == 0x00040000)      //Check Data Read Pin SDA(PA0) is '0' or '1'
	    result |= 0x01                      ;   //If Read Pin SDA is '1' Give Save Bit Data = 1

      SCL_LO()                              ;   //Next Bit Read

    }
	 
  return result                             ;

}



/*************************************************************
 **              Function Read RTC(DS3232) 1 Byte           **
 ** Parameter :                                             **
 **              addr = Register Address for Read            **
 *************************************************************/

 

 signed char I2C_RdRTC(unsigned char addr)
  {
    signed char dat  ;
   
 //-------------- Sent Status Start for write -----------

   I2C_Start();

//------------- Sent Slave Address for Write------------	
  
   I2C_WrByte(0xD0)        ;				 //Send ID Code DS3232,Write (1101 000+0)

//------------ Sent address Register --------------

   I2C_WrByte(addr)       ;					//Send Address Reg. to DS3232
 
//---------- New Sent Status Start For Read ----------

   I2C_Start()            ;

//-------------Sent Slave address for Read ------------

  I2C_WrByte(0xD1)       ;					//Send ID Code DS3232 ,Write (1101 000+1)

//------------- Read data 1 Byte ------------
   
   dat = I2C_RdByte()       ;					
  
  

//---------- Sent Status Stop Red ---------------
   I2C_Stop(); 
 
	return dat            ;                 //Return Data
  }




/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  +                                                                            +
  +                      Function Convert data to ASCII                        +
  +                                                                            +
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/


/**************************************************************
 *															  *
 *                  Convert BCD Time to ascii                 *
 *                                                            *
 **************************************************************/

 void conv_time(char sec,char min,char hour)
 {
 
   //---------- Convert Second(00-59) to ascii ---------
   ss  = sec >> 4             ;
   ss  = ss+0x30  		      ; //Secound byte High to ascii

   sec = sec & 0x0F           ;
   sec = sec+0x30             ; //Secound byte Low to ascii
   ss  = (ss << 8)| sec       ; 

   //---------- Convert Minute(00-59) to ascii ---------

   mn  = min >> 4             ;
   mn  = mn+0x30  		      ;  //Minuted byte High to ascii

   min = min & 0x0F           ;
   min = min+0x30             ;  //Minuted byte Low to ascii
   mn  = (mn << 8)| min       ; 

   //---------- Convert Hour(00-23) to ascii ---------

   hh   = hour >> 4           ;  //Hour byte High to ascii
   hh   = hh+0x30  		      ;

   hour = hour & 0x0F         ;
   hour = hour+0x30           ;  //Hour byte Low to ascii
   hh   = (hh << 8)| hour     ; 

 }

 
/**************** *********************************************
 *				Convert BCD DATE to ascii 					  *
 **************************************************************/

 void conv_date(char date,char month,char year)
 {
  
   //---------- Convert date(00-31) to ascii ---------
   dd   = date >> 4             ;
   dd   = dd+0x30  		        ;  //Date byte High to ascii

   date = date & 0x0F           ;
   date = date+0x30             ;  //Date byte Low to ascii
   dd   = (dd << 8)| date       ; 

   //---------- Convert Month(00-12) to ascii ---------

   mm    = (month >> 4)& 0x07   ;  //Clear bit Century
   mm    = mm+0x30  		    ;  //Month byte High to ascii

   month = month & 0x0F         ;
   month = month+0x30           ;  //Month byte Low to ascii
   mm    = (mm << 8)| month     ; 

   //---------- Convert Year(00-99) to ascii ---------

   yy   = year >> 4             ;  //Year byte High to ascii
   yy   = yy+0x30  		        ;

   year = year & 0x0F           ;
   year = year+0x30             ;  //Year byte Low to ascii
   yy   = (yy << 8)| year       ; 

 }
 
/**************************************************************
 *				Convert hex Temp to ascii				      *
 **************************************************************/

void conv_temp(signed char tm,unsigned char tf)
{

  //------------- Sign to ascii -------------
	if(tm & 0x80)
	  sign = '-'                        ;	   //Temp- show sign '-'
    else
	  sign = 0x20                       ;	   //Temp+ show sign space

  //------------- Temp. Integer hex to ascii -------------

    tm  = tm & 0x7F                     ;      //Clear bit sign
    tem = (tm/10)+0x30	                ;      //Keep Temp byte high to Ascii     
	tem = (tem <<8)|((tm%10)+0x30)      ;	   //Keep Temp byte Low to Ascii 
 
  //------------- Temp.(Step=0.25C) fractional hex to ascii -------------

  	tf  = tf >>6                        ;     //Shift data bit 7..6  to bit  1..0    
	tf  = tf*25                         ;     //Step temp 25
	dot = (tf/10)+0x30                  ;     //hex to Ascii byte High
	dot = (dot<<8)|((tf%10)+0x30)       ; 	  //hex to Ascii byte Low
} 


 
/* ################################################################
   ##														     ##
   ##                        Main Program                        ##
   ##														 	 ##
   ################################################################ */
 
int main(void)
   {  
	 unsigned char sec,min,hor,tf,date,month,year ;
	 unsigned char tp      ;
	 unsigned int n        ;


    init_port()			   ;
	init_lcd()             ;    

 
//------------ write string to lcd ----------------
  
  //----------- String mode Time ----------

   write_cmm(0x80|0x00)    ;  //Start address lcd = 0x00  Line 1
   write_data('T')         ;  //Plot data to lcd
   write_data('i')         ;
   write_data('m')         ;
   write_data('e')         ;
   write_data(':')         ;
   write_cmm(0x80|0x07)    ;  //Start address lcd = 0x07  Line 1
   write_data(':')         ;
   write_cmm(0x80|0x0A)    ;  //Start address lcd = 0x0A  Line 1
   write_data(':')         ;

 
 
   delay(100);
 
 
//--------- Set Time Secound:Minute:Hour to DS3232-------------

   I2C_WrRTC(0x00,0x00)    ;  //Write Second   
   I2C_WrRTC(0x01,0x00)    ;  //Write minute
   I2C_WrRTC(0x02,0x12)    ;  //Write Hour

//--------- Set Date Date:Month:Year to DS3232-------------

   I2C_WrRTC(0x04,0x01)    ;  //Write Date
   I2C_WrRTC(0x05,0x01)    ;  //Write Month
   I2C_WrRTC(0x06,0x12)    ;  //Write year
  
  while(1)
  {

//--------------------- Display Time & Temp ----------------------

    //----------- String mode Temp ----------

   write_cmm(0x80|0x40)    ;  //Start address lcd = 0x40 Line 2
   write_data('T')         ;  //Plot data to lcd
   write_data('e')         ;
   write_data('m')         ;
   write_data('p')         ;
   write_data(':')         ;
   write_cmm(0x80|0x45)    ;  //Start address lcd = 0x45 Line 2
   write_data(0x20)        ;  //Plot data to lcd for sige -
   
   write_cmm(0x80|0x48)    ;  //Start address lcd = 0x48 Line 2
   write_data('.')         ;  //Plot data to lcd
 

   write_cmm(0x80|0x4B)    ;  //Start address lcd = 0x4B Line 2
   write_data(0xD2)        ;  //Plot data to lcd
   write_data('C')         ;  //Plot data to lcd


   for(n=0;n<300;n++)
	{

 //------------ Read Time ss:mm:hh from ds3232 --------------

 	 sec = I2C_RdRTC(0x00)          ;  //Read Second
     min = I2C_RdRTC(0x01)          ;  //Read minute
	 hor = I2C_RdRTC(0x02)          ;  //Read Hour  24 Hour
	 conv_time(sec,min,hor)         ;  //Convert sec:min:hor to ascii
	
//------------ Read Temp from ds3232 --------------

	 tp = I2C_RdRTC(0x11)           ;  //Read interger + sige temp. 
     tf = I2C_RdRTC(0x12)           ;  //Read fractional(step =0.25 C) temp.
	 conv_temp(tp,tf)               ;


//------------ Write Data hh:mm:ss to LCD -------------

 	 write_cmm(0x80|0x05)    ;  //Start address lcd = 0x05
     write_data(hh>>8)       ;
	 write_data(hh)          ;
	
	 write_cmm(0x80|0x08)    ;  //Start address lcd = 0x08
     write_data(mn>>8)       ;
	 write_data(mn)          ;

	 write_cmm(0x80|0x0B)    ;  //Start address lcd = 0x0B
     write_data(ss>>8)       ;
	 write_data(ss)          ;

//------------ Write Data Temp to LCD -------------

  	 write_cmm(0x80|0x45)    ;  //Start address lcd = 0x48
     write_data(sign)        ;  // data Sign

     write_data(tem>>8)      ;  //data integer Temp
	 write_data(tem)         ;

	 write_cmm(0x80|0x49)    ;  //Start address lcd = 0x4B
     write_data(dot>>8)      ;  //data Fractional Temp
	 write_data(dot)         ;

   }



  //-----------------Display Time & Date ------------------


	 //----------- String mode Date ----------

   write_cmm(0x80|0x40)    ;  //Start address lcd = 0x40 Line 2
   write_data('D')         ;  //Plot data to lcd
   write_data('a')         ;
   write_data('t')         ;
   write_data('e')         ;
   write_data(':')         ;
   write_cmm(0x80|0x47)    ;  //Start address lcd = 0x47 Line 2
   write_data('/')         ;  //Plot data to lcd for sige -
   
   write_cmm(0x80|0x4A)    ;  //Start address lcd = 0x4A Line 2
   write_data('/')         ;  //Plot data to lcd
	

   for(n=0;n<300;n++)
	{

 //------------ Read Time ss:mm:hh from ds3232 --------------

	  sec = I2C_RdRTC(0x00)          ;  //Read Second
      min = I2C_RdRTC(0x01)          ;  //Read minute
	  hor = I2C_RdRTC(0x02)          ;  //Read Hour  24 Hour
	  conv_time(sec,min,hor)         ;  //Convert sec:min:hor to ascii

//------------ Read Date:Mount:Year  from ds3232 --------------
	  date  = I2C_RdRTC(0x04)        ;  //Read date (1-31). 
      month = I2C_RdRTC(0x05)        ;  //Read Mount
	  year  = I2C_RdRTC(0x06)        ;  //Read year
	  conv_date(date,month,year)     ;  //


//------------ Write Data hh:mn:ss to LCD -------------

	  write_cmm(0x80|0x05)    ;  //Start address lcd = 0x05
      write_data(hh>>8)       ;
	  write_data(hh)          ;
	
	  write_cmm(0x80|0x08)    ;  //Start address lcd = 0x08
      write_data(mn>>8)       ;
	  write_data(mn)          ;

	  write_cmm(0x80|0x0B)    ;  //Start address lcd = 0x0B
      write_data(ss>>8)       ;
	  write_data(ss)          ;

//------------ Write Data dd:mm:yy to LCD -------------

  	  write_cmm(0x80|0x45)    ;  //Start address lcd = 0x45
      write_data(dd>>8)       ;  //Plot data date
	  write_data(dd)          ;

	  write_cmm(0x80|0x48)    ;  //Start address lcd = 0x48
      write_data(mm>>8)       ;  //Plot data month
	  write_data(mm)          ;
  
      write_cmm(0x80|0x4B)    ;  //Start address lcd = 0x4B
      write_data(yy>>8)       ;  //Plot data year
	  write_data(yy)          ;
   
   }
 }

}

